﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public abstract class MapperBase
    {
        public Guid NetworkId { get; set; }

        public EntityReference Owner { get; set; }

        public PpmsReference References { get; set; }

        public string NetworkShorthand { get; set; }

        public const string NetworkRelationship = "VA Provider";

        protected bool IsWithinContext { get; set; }
        public bool ForVaNetwork { get; set; }

        protected readonly IPpmsContextHelper PpmsContextHelper;
        protected readonly IPpmsHelper PpmsHelper;

        private PpmsContext _context;
        protected PpmsContext Context
        {
            get
            {
                return _context;
            }
            set
            {
                _context = value;
                // If context is set, use provided context for update activities
                IsWithinContext = _context != null;
            }
        }

        protected MapperBase(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
        {
            PpmsContextHelper = ppmsContextHelper;
            PpmsHelper = ppmsHelper;
        }
        
        public IEnumerable<SetStateRequest> EntityDelete<T>(IEnumerable<T> list)
        {
            var requests = new List<SetStateRequest>();
            var items = list as T[] ?? list.ToArray();
            if (!items.Any()) return requests;

            requests.AddRange(items.Select(EntityDelete).Where(request => request != null));

            return requests;
        }

        public SetStateRequest EntityDelete<T>(T entity)
        {
            return entity != null ? DeactivateEntity(entity) : null;
        }

        public void ResetContext()
        {
            if (Context != null) Context.Dispose();
            Context = null;
        }

        protected SetStateRequest DeactivateEntity<T>(T entity)
        {
            // Set default status code for all entities
            int statusCode = (int)Account_StatusCode.Inactive;
            // If entity is Account, set status code to Deactivate
            var account = entity as Entity;
            if (account != null && account.LogicalName == "account")
            {
                statusCode = (int)Account_StatusCode.Deactivated;
            }

            return PpmsHelper.SetEntityStatus(entity, (int)AccountState.Inactive, statusCode);
        }

        protected SetStateRequest ActivateEntity<T>(T entity)
        {
            return PpmsHelper.SetEntityStatus(entity, (int)AccountState.Active, (int)Account_StatusCode.Active);
        }

        protected bool IsChanged(string source, string target)
        {
            return !string.IsNullOrEmpty(source) && source != target;
        }

        protected void HandleException(string message)
        {
            throw new PpmsServiceException(message);
        }

        protected void HandleException(string message, string id)
        {
            HandleException($"{message} [Entity Id: {id}]");
        }

        protected IEnumerable<T> ConvertEntityList<T>(IEnumerable<object> entityList)
        {
            var objects = entityList as object[] ?? entityList.ToArray();
            if (!objects.Any()) return new List<T>();
            var list = objects.ToList();

            return list.ConvertAll(x => (T)x);
        }

        protected void SetNetworkByShorthand(string shorthand)
        {
            if (string.IsNullOrEmpty(shorthand)) return;

            var network = GetNetwork(shorthand);
            if (network != null)
            {
                NetworkId = network.Id;
            }
        }

        protected async Task<Account> GetProvider(Provider provider)
        {
            if (provider != null)
            {
                return string.IsNullOrEmpty(provider.CorrelationId)
                    ? await GetProviderByNpi(provider.FirstNpi.Number, true) as Account
                    : await GetProviderByCorrelationId(provider.CorrelationId, true) as Account;
            }
            else return null;
        }

        protected async Task<Entity> GetProviderByCorrelationId(string correlationId, bool loadProperties = false)
        {
            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                var id = new Guid(correlationId);
                var entity = context.AccountSet.FirstOrDefault(a => a.AccountId == id);
                if (loadProperties) LoadProviderProperties(entity, context);

                return entity;
            }
        }

        protected async Task<Entity> GetProviderByNpi(string npi, bool loadProperties = false)
        {
            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                var provider = context.AccountSet.FirstOrDefault(a => a.ppms_ProviderIdentifier == npi);
                if (loadProperties) LoadProviderProperties(provider, context);

                return provider;
            }
        }

        protected async Task<va_visn> GetVisn(string visn)
        {
            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                return context.va_visnSet.FirstOrDefault(s => s.va_name == visn);
            }
        }

        #region PPMS References

        protected EntityReference GetState(string state)
        {
            if (References != null && References.States != null)
            {
                var result = References.States.FirstOrDefault(s => s.Value == state.Trim());
                return new EntityReference("va_state", result.Id);
            }

            HandleException($"GetState: Unable to retrieve reference ({state}).");
            return null;
        }

        protected EntityReference GetTaxonomy(string specialty)
        {
            Int16 state = 0;

            if (References != null && References.Specialties != null)
            {
                state++;
                var result = References.Specialties.FirstOrDefault(s => s.Value == specialty.Trim());
                if (result != null)
                    return new EntityReference("ppms_taxonomy", result.Id);
            }

            var debugMsg = new StringBuilder();
            if (References == null) debugMsg.Append("Null References; ");
            else if (References.Specialties == null) debugMsg.Append("Null Specialties; ");
            else debugMsg.Append($"Specialties: {References.Specialties.Count}; ");

            HandleException($"GetTaxonomy: Unable to retrieve taxonomy reference ({specialty},{state}). {debugMsg.ToString()}");
            return null;
        }

        protected EntityReference GetNetwork(string shorthand, string relationship)
        {
            string networkName = string.IsNullOrEmpty(relationship) ? $"{shorthand}" : $"{shorthand}_{relationship}";
            return GetNetwork(networkName);
        }

        protected EntityReference GetNetwork(string shorthand)
        {
            if (References != null && References.Networks != null)
            {
                var result = References.Networks.FirstOrDefault(s => s.Value == shorthand.Trim());
                if (result != null)
                    return new EntityReference("ppms_vaprovidernetwork", result.Id);
            }

            HandleException($"GetNetwork: Unable to retrieve reference ({shorthand}).");
            return null;
        }

        protected IEnumerable<EntityReference> GetNetworkGroup(string shorthand)
        {
            if (References != null && References.Networks != null)
            {
                var result = References.Networks.Where(s => s.Value.Contains(shorthand.Trim()));
                if (result != null)
                {
                    return result.ToList().ConvertAll(x => new EntityReference("ppms_vaprovidernetwork", x.Id));
                }
            }

            HandleException($"GetNetworkGroup: Unable to retrieve references ({shorthand}).");
            return null;
        }

        protected EntityReference GetRelationship(string name)
        {
            if (References != null && References.VaProviderRelationships != null)
            {
                var result = References.VaProviderRelationships.FirstOrDefault(s => s.Value == name.Trim());
                if (result != null)
                    return new EntityReference("ppms_vaproviderrelationship", result.Id);
            }

            HandleException($"GetRelationship: Unable to retrieve reference ({name}).");
            return null;
        }

        protected EntityReference GetPlaceOfServiceCode(string name)
        {
            if (References != null && References.PlaceOfServiceCodes != null)
            {
                var result = References.PlaceOfServiceCodes.FirstOrDefault(s => s.Value == name.Trim());
                if (result != null)
                    return new EntityReference("ppms_placeofservicecode", result.Id);
            }

            HandleException($"GetPlaceOfServiceCode: Unable to retrieve reference ({name}).");
            return null;
        }

        protected EntityReference GetFacilityByStationNumber(string name)
        {
            if (References != null && References.FacilitiesByStation != null)
            {
                var result = References.FacilitiesByStation.FirstOrDefault(s => s.Value == name.Trim());
                if (result != null)
                    return new EntityReference("va_facility", result.Id);
            }

            HandleException($"GetFacilitiesByStationNumber: Unable to retrieve reference ({name}).");
            return null;
        }

        #endregion

        protected void LoadProviderProperties(Entity entity, PpmsContext context)
        {
            if (entity == null || context == null) return;

            context.LoadProperty(entity, "ppms_account_ppms_provideridentifier_Provider");
            context.LoadProperty(entity, "ppms_account_ppms_providerservice");
            context.LoadProperty(entity, "ppms_account_ppms_boardcertification");
            context.LoadProperty(entity, "ppms_account_organizationauthorizedofficial");
            context.LoadProperty(entity, "ppms_account_ppms_otherprovideridentifier");
            context.LoadProperty(entity, "contact_customer_accounts");
            context.LoadProperty(entity, "ppms_account_providerlicensure");
            context.LoadProperty(entity, "ppms_account_ppms_othername");
            context.LoadProperty(entity, "ppms_account_ppms_providertaxonomy");
            context.LoadProperty(entity, "ppms_account_deascheduleprivilege");
            context.LoadProperty(entity, "ppms_account_providernetworkid");
            context.LoadProperty(entity, "ppms_account_caresite_organization");
        }

        #region References

        public async Task ResetPpmsReferences()
        {
            References = await PpmsHelper.GetPpmsReferences();
        }

        public void SetPpmsReferences(PpmsReference references)
        {
            References = references;
        }

        #endregion

        protected DateTime? ToUtcDate(DateTime dateTime)
        {
            return new DateTime(dateTime.Year, dateTime.Month, dateTime.Day, dateTime.Hour, dateTime.Minute, dateTime.Second, DateTimeKind.Local);
        }
    }
}